// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_BLINK_RENDERER_MODULES_WEBGPU_GPU_TEXTURE_H_
#define THIRD_PARTY_BLINK_RENDERER_MODULES_WEBGPU_GPU_TEXTURE_H_

#include "third_party/blink/renderer/modules/webgpu/dawn_object.h"
#include "third_party/blink/renderer/platform/wtf/ref_counted.h"

namespace blink {

class ExceptionState;
class GPUTextureDescriptor;
class GPUTextureView;
class GPUTextureViewDescriptor;
class StaticBitmapImage;
class WebGPUMailboxTexture;

class GPUTexture : public DawnObject<WGPUTexture> {
  DEFINE_WRAPPERTYPEINFO();

 public:
  static GPUTexture* Create(GPUDevice* device,
                            const GPUTextureDescriptor* webgpu_desc,
                            ExceptionState& exception_state);
  static GPUTexture* CreateError(GPUDevice* device,
                                 const WGPUTextureDescriptor* desc);

  GPUTexture(GPUDevice* device, WGPUTexture texture, const String& label);
  GPUTexture(GPUDevice* device,
             WGPUTextureFormat format,
             WGPUTextureUsage usage,
             scoped_refptr<WebGPUMailboxTexture> mailbox_texture,
             const String& label);

  ~GPUTexture() override;

  GPUTexture(const GPUTexture&) = delete;
  GPUTexture& operator=(const GPUTexture&) = delete;

  // gpu_texture.idl
  GPUTextureView* createView(const GPUTextureViewDescriptor* webgpu_desc,
                             ExceptionState& exception_state);
  void destroy();
  uint32_t width() const;
  uint32_t height() const;
  uint32_t depthOrArrayLayers() const;
  uint32_t mipLevelCount() const;
  uint32_t sampleCount() const;
  String dimension() const;
  String format() const;
  uint32_t usage() const;

  WGPUTextureDimension Dimension() { return dimension_; }
  WGPUTextureFormat Format() { return format_; }
  WGPUTextureUsageFlags Usage() { return usage_; }
  bool Destroyed() { return destroyed_; }

  void DissociateMailbox();

  // Returns a shared pointer to the mailbox texture. The mailbox texture
  // remains associated to the GPUTexture.
  scoped_refptr<WebGPUMailboxTexture> GetMailboxTexture();

  // Sets a callback which is called if destroy is called manually, before the
  // WebGPU handle is actually destroyed.
  void SetBeforeDestroyCallback(base::OnceClosure);
  void ClearBeforeDestroyCallback();

 private:
  void setLabelImpl(const String& value) override {
    std::string utf8_label = value.Utf8();
    GetProcs().textureSetLabel(GetHandle(), utf8_label.c_str());
  }

  WGPUTextureDimension dimension_;
  WGPUTextureFormat format_;
  WGPUTextureUsageFlags usage_;
  scoped_refptr<WebGPUMailboxTexture> mailbox_texture_;
  bool destroyed_ = false;
  base::OnceClosure destroy_callback_;
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_MODULES_WEBGPU_GPU_TEXTURE_H_
